#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2019-2026 -- leonerd@leonerd.org.uk

package Future::IO 0.21;

use v5.14;
use warnings;

use Carp;

# These need to be visible to sub override_impl
my @pollers;
my @alarms;

our $IMPL;

our $MAX_READLEN = 8192;
our $MAX_WRITELEN = 8192;

use IO::Poll qw( POLLIN POLLOUT POLLPRI POLLHUP POLLERR POLLNVAL );

use Exporter 'import';
BEGIN {
   # This needs to happen at BEGIN time because stupid cyclic reasons
   our @EXPORT_OK = qw( POLLIN POLLOUT POLLPRI POLLHUP POLLERR POLLNVAL );
}

=head1 NAME

C<Future::IO> - Future-returning IO methods

=head1 SYNOPSIS

=for highlighter language=perl

   use Future::IO;
   Future::IO->load_best_impl;

   my $delay = Future::IO->sleep( 5 );
   # $delay will become done in 5 seconds time

   my $input = Future::IO->read( \*STDIN, 4096 );
   # $input will yield some input from the STDIN IO handle

=head1 DESCRIPTION

This package provides a few basic methods that behave similarly to the
same-named core perl functions relating to IO operations, but yield their
results asynchronously via L<Future> instances.

This is provided primarily as a decoupling mechanism, to allow modules to be
written that perform IO in an asynchronous manner to depend directly on this,
while allowing asynchronous event systems to provide an implementation of
these operations.

=head2 Default Implementation

If the C<override_impl> method is not invoked, a default implementation of
these operations is provided. This implementation allows a single queue of
C<read> or C<write> calls on a single filehandle only, combined with C<sleep>
calls. It does not support C<waitpid>.

It is provided for the simple cases where modules only need one filehandle
(most likely a single network socket or hardware device handle), allowing such
modules to work without needing a better event system.

If there are both read/write and C<sleep> futures pending, the implementation
will use C<select()> to wait for either. This may be problematic on MSWin32,
depending on what type of filehandle is involved.

If C<select()> is not being used then the default implementation will
temporarily set filehandles into blocking mode (by switching off the
C<O_NONBLOCK> flag) while performing IO on them.

For cases where multiple filehandles are required, or for doing more involved
IO operations, a real implementation based on an actual event loop should be
loaded. It is recommended to use the L</load_best_impl> method to do this, if
there are no other specific requirements. If the program is already using some
other event system, such as L<UV> or L<IO::Async>, it is best to directly load
the relevant implementation module in the toplevel program.

=head2 Unit Testing

The replaceable implementation is also useful for writing unit test scripts.
If the implementation is set to an instance of some sort of test fixture or
mocking object, a unit test can check that the appropriate IO operations
happen as part of the test.

A testing module which does this is provided by L<Test::Future::IO>.

=head2 Cancellation

Any C<Future> returned by a C<Future::IO> method should support being
cancelled by the L<Future/cancel> method. Doing so should not cause the
program to break overall, nor will it upset the specific implementation being
used.

I<However>, the result of cancelling a future instance that performs some
actual IO work is I<unspecified>. It may cause no work to be performed, or it
may result in a partial (or even complete but as-yet unreported) IO operation
to have already taken place. In particular, operations that write bytes to, or
read bytes from filehandles may have already transferred some or all of those
bytes before the future was cancelled, and there is now nothing that the
program can do to "undo" those effects. In general it is likely that the only
situation where you will cancel an IO operation on a filehandle is when an
entire connection is being abandoned, filehandles closed, and so on.

That said, it should be safe to cancel L</alarm> and L</sleep> futures, as
each one will operate entirely independently, and not cause any change of
state to any of the others. This typically allows you to wrap a "timeout"-like
behaviour around any other sort of IO operation by using L<Future/needs_any>
or similar. If the real IO operation was successful, the timeout can be safely
cancelled. If the timeout happens, the IO operation will be cancelled, and at
this point the application will have to discard the filehandle (or otherwise
resynchronse in some application-specific manner).

=cut

=head1 METHODS

=cut

=head2 accept

   $socketfh = await Future::IO->accept( $fh );

I<Since version 0.11.>

Returns a L<Future> that will become done when a new connection has been
accepted on the given filehandle, which should represent a listen-mode socket.
The returned future will yield the newly-accepted client socket filehandle.

=cut

sub accept
{
   shift;
   my ( $fh ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->accept( $fh );
}

=head2 alarm

   await Future::IO->alarm( $epoch );

I<Since version 0.08.>

Returns a L<Future> that will become done at a fixed point in the future,
given as an epoch timestamp (such as returned by C<time()>). This value may be
fractional.

=cut

sub alarm
{
   shift;
   my ( $epoch ) = @_;

   $IMPL //= "Future::IO::_DefaultImpl";

   if( $IMPL->can( "alarm" ) ) {
      return $IMPL->alarm( $epoch );
   }
   else {
      return $IMPL->sleep( $epoch - Time::HiRes::time() );
   }
}

=head2 connect

   await Future::IO->connect( $fh, $name );

I<Since version 0.11.>

Returns a L<Future> that will become done when a C<connect()> has succeeded on
the given filehandle to the given sockname address.

=cut

sub connect
{
   shift;
   my ( $fh, $name ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->connect( $fh, $name );
}

=head2 poll

   $revents = await Future::IO->poll( $fh, $events );

I<Since version 0.19.>

Returns a L<Future> that will become done when the indicated IO operations
can be performed on the given filehandle. I<$events> should be a bitfield of
one or more of the POSIX C<POLL*> constants, such as C<POLLIN>, C<POLLOUT> or
C<POLLPRI>. The result of the future will be a similar bitfield, indicating
which operations may now take place. If the C<POLLHUP>, C<POLLERR> or
C<POLLNVAL> events happen, they will always be reported; you do not need to
request these specifically.

Multiple outstanding futures may be enqueued for the same filehandle. When an
event happens, only the first outstanding future that is interested in it is
informed; the rest will remain pending for the next round of IO events, if the
condition still prevails.

Note that, as compared to the real C<poll(2)> system call, this method only
operates on a single filehandle; all futures returned by it are independent
and refer just to that one filehandle each.

Also note that, in general, it is better to use one of the higher-level
methods to perform whatver IO operation is required on the given filehandle.
The C<poll> method is largely intended as a lowest-level fallback, for example
if integrating with some other library or module that performs its own
filehandle IO and just needs to be informed when such IO operations may be
performed.

For convenience, the C<POLL*> constants are exported by this module. They
should be used in preference to the ones from C<IO::Poll>, in case a platform
does not provide the latter module directly.

=cut

sub poll
{
   shift;
   my ( $fh, $events ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->poll( $fh, $events );
}

=head2 read

   $bytes = await Future::IO->read( $fh, $length );

I<Since version 0.17.> Before this version this method used to be named
C<sysread>, and still available via that alias.

Returns a L<Future> that will become done when at least one byte can be read
from the given filehandle. It may return up to C<$length> bytes. On EOF, the
returned future will yield an empty list (or C<undef> in scalar context). On
any error (other than C<EAGAIN> / C<EWOULDBLOCK> which are ignored), the
future fails with a suitable error message.

Note specifically this may perform only a single C<sysread()> call, and thus
is not guaranteed to actually return the full length.

=cut

sub read
{
   shift;
   my ( $fh, $length ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->sysread( $fh, $length );
}

*sysread = \&read;

=head2 read_exactly

   $bytes = await Future::IO->read_exactly( $fh, $length );

I<Since version 0.17.> Before this version this method used to be named
C<sysread_exactly>, and still available via that alias.

Returns a L<Future> that will become done when exactly the given number of
bytes have been read from the given filehandle. It returns exactly C<$length>
bytes. On EOF, the returned future will yield an empty list (or C<undef> in
scalar context), even if fewer bytes have already been obtained. These bytes
will be lost. On any error (other than C<EAGAIN> / C<EWOULDBLOCK> which are
ignored), the future fails with a suitable error message.

This may make more than one C<sysread()> call.

=cut

sub read_exactly
{
   shift;
   my ( $fh, $length ) = @_;

   $IMPL //= "Future::IO::_DefaultImpl";

   if( my $code = $IMPL->can( "sysread_exactly" ) ) {
      return $IMPL->$code( $fh, $length );
   }

   return _read_into_buffer( $IMPL, $fh, $length, \(my $buffer = '') );
}

*sysread_exactly = \&read_exactly;

sub _read_into_buffer
{
   my ( $IMPL, $fh, $length, $bufref ) = @_;

   $IMPL->sysread( $fh, $length - length $$bufref )->then( sub {
      my ( $more ) = @_;
      return Future->done() if !defined $more; # EOF

      $$bufref .= $more;

      return Future->done( $$bufref ) if length $$bufref >= $length;
      return _read_into_buffer( $IMPL, $fh, $length, $bufref );
   });
}

=head2 read_until_eof

   $f = Future::IO->read_until_eof( $fh );

I<Since version 0.17.> Before this version this method used to be named
C<sysread_until_eof>, and still available via that alias.

Returns a L<Future> that will become done when the given filehandle reaches
the EOF condition. The returned future will yield all of the bytes read up
until that point.

=cut

sub read_until_eof
{
   shift;
   my ( $fh ) = @_;

   $IMPL //= "Future::IO::_DefaultImpl";

   return _read_until_eof( $IMPL, $fh, \(my $buffer = '') );
}

*sysread_until_eof = \&read_until_eof;

sub _read_until_eof
{
   my ( $IMPL, $fh, $bufref ) = @_;

   $IMPL->sysread( $fh, $MAX_READLEN )->then( sub {
      my ( $more ) = @_;
      return Future->done( $$bufref ) if !defined $more;

      $$bufref .= $more;
      return _read_until_eof( $IMPL, $fh, $bufref );
   });
}

=head2 recv

=head2 recvfrom

   $bytes = await Future::IO->recv( $fh, $length );
   $bytes = await Future::IO->recv( $fh, $length, $flags );

   ( $bytes, $from ) = await Future::IO->recvfrom( $fh, $length );
   ( $bytes, $from ) = await Future::IO->recvfrom( $fh, $length, $flags );

I<Since version 0.17.>

Returns a L<Future> that will become done when at least one byte is received
from the given filehandle (presumably a socket), by using a C<recv(2)> or
C<recvfrom(2)> system call. On any error (other than C<EAGAIN> /
C<EWOULDBLOCK> which are ignored) the future fails with a suitable error
message.

Optionally a flags bitmask in C<$flags> can be passed. If no flags are
required, the value may be zero. The C<recvfrom> method additionally returns
the sender's address as a second result value; this is primarily used by
unconnected datagram sockets.

=cut

sub recv
{
   shift;
   my ( $fh, $length, $flags ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->recv( $fh, $length, $flags );
}

sub recvfrom
{
   shift;
   my ( $fh, $length, $flags ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->recvfrom( $fh, $length, $flags );
}

=head2 send

   $sent_len = await Future::IO->send( $fh, $bytes );
   $sent_len = await Future::IO->send( $fh, $bytes, $flags );
   $sent_len = await Future::IO->send( $fh, $bytes, $flags, $to );

I<Since version 0.17.>

Returns a L<Future> that will become done when at least one byte has been
sent to the given filehandle (presumably a socket), by using a C<send(2)>
system call. On any error (other than C<EAGAIN> / C<EWOULDBLOCK> which are
ignored) the future fails with a suitable error message.

Optionally a flags bitmask in C<$flags> or a destination address in C<$to> can
also be passed. If no flags are required, the value may be zero. If C<$to> is
specified then a C<sendto(2)> system call is used instead.

=cut

sub send
{
   shift;
   my ( $fh, $bytes, $flags, $to ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->send( $fh, $bytes, $flags, $to );
}

=head2 sleep

   await Future::IO->sleep( $secs );

Returns a L<Future> that will become done a fixed delay from now, given in
seconds. This value may be fractional.

=cut

sub sleep
{
   shift;
   my ( $secs ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->sleep( $secs );
}

=head2 waitpid

   $wstatus = await Future::IO->waitpid( $pid );

I<Since version 0.09.>

Returns a L<Future> that will become done when the given child process
terminates. The future will yield the wait status of the child process.
This can be inspected by the usual bitshifting operations as per C<$?>:

   if( my $termsig = ($wstatus & 0x7f) ) {
      say "Terminated with signal $termsig";
   }
   else {
      my $exitcode = ($wstatus >> 8);
      say "Terminated with exit code $exitcode";
   }

=cut

sub waitpid
{
   shift;
   my ( $pid ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->waitpid( $pid );
}

=head2 write

   $written_len = await Future::IO->write( $fh, $bytes );

I<Since version 0.17.> Before this version this method used to be named
C<syswrite>, and still available via that alias.

Returns a L<Future> that will become done when at least one byte has been
written to the given filehandle. It may write up to all of the bytes. On any
error (other than C<EAGAIN> / C<EWOULDBLOCK> which are ignored) the future
fails with a suitable error message.

Note specifically this may perform only a single C<syswrite()> call, and thus
is not guaranteed to actually return the full length.

=cut

sub write
{
   shift;
   my ( $fh, $bytes ) = @_;

   return ( $IMPL //= "Future::IO::_DefaultImpl" )->syswrite( $fh, $bytes );
}

*syswrite = \&write;

=head2 write_exactly

   $written_len = await Future::IO->write_exactly( $fh, $bytes );

I<Since version 0.17.> Before this version this method used to be named
C<syswrite_exactly>, and still available via that alias.

Returns a L<Future> that will become done when exactly the given bytes have
been written to the given filehandle. On any error (other than C<EAGAIN> /
C<EWOULDBLOCK> which are ignored) the future fails with a suitable error
message.

This may make more than one C<syswrite()> call.

=cut

sub write_exactly
{
   shift;
   my ( $fh, $bytes ) = @_;

   $IMPL //= "Future::IO::_DefaultImpl";

   if( my $code = $IMPL->can( "syswrite_exactly" ) ) {
      return $IMPL->$code( $fh, $bytes );
   }

   return _write_from_buffer( $IMPL, $fh, \$bytes, length $bytes );
}

*syswrite_exactly = \&write_exactly;

sub _write_from_buffer
{
   my ( $IMPL, $fh, $bufref, $len ) = @_;

   $IMPL->syswrite( $fh, substr $$bufref, 0, $MAX_WRITELEN )->then( sub {
      my ( $written_len ) = @_;
      substr $$bufref, 0, $written_len, "";

      return Future->done( $len ) if !length $$bufref;
      return _write_from_buffer( $IMPL, $fh, $bufref, $len );
   });
}

=head2 override_impl

   Future::IO->override_impl( $impl );

Sets a new implementation for C<Future::IO>, replacing the minimal default
internal implementation. This can either be a package name or an object
instance reference, but must provide the methods named above.

This method is intended to be called by event loops and other similar places,
to provide a better integration. Another way, which doesn't involve directly
depending on C<Future::IO> or loading it, is to use the C<$IMPL> variable; see
below.

Can only be called once, and only if the default implementation is not in use,
therefore a module that wishes to override this ought to invoke it as soon as
possible on program startup, before any of the main C<Future::IO> methods may
have been called.

=cut

my $overridden;

sub override_impl
{
   shift;
   croak "Future::IO implementation is already overridden" if defined $IMPL;
   croak "Future::IO implementation cannot be set once default is already in use"
      if @pollers or @alarms;

   ( $IMPL ) = @_;
}

sub try_load_impl
{
   shift;
   my ( $name ) = @_;

   $name =~ m/::/ or $name = "Future::IO::Impl::$name";
   my $module = "$name.pm" =~ s{::}{/}gr;

   eval { require $module } or return 0;
   return 1;
}

=head2 load_impl

   Future::IO->load_impl( @names );

I<Since version 0.16.>

Given a list of possible implementation module names, iterates through them
attempting to load each one until a suitable module is found. Any errors
encountered while loading each are ignored. If no module is found to be
suitable, an exception is thrown that likely aborts the program.

C<@names> should contain a list of Perl module names (which likely live in the
C<Future::IO::Impl::*> prefix). If any name does not contain a C<::>
separator, it will have that prefix applied to it. This allows a conveniently
short list; e.g.

   Future::IO->load_impl( qw( UV Glib IOAsync ) );

This method is intended to be called once, at startup, by the main containing
program. Since it sets the implementation, it would generally be considered
inappropriate to invoke this method from some additional module that might be
loaded by a containing program.

This is now discouraged, in favour of letting C<Future::IO> decide instead by
using L</load_best_impl>.

=cut

sub load_impl
{
   shift;

   foreach ( @_ ) {
      Future::IO->try_load_impl( $_ ) and return;
   }
   die "Unable to find a usable Future::IO::Impl subclass\n";
}

=head2 load_best_impl

   Future::IO->load_best_impl();

I<Since version 0.18.>

Attempt to work out and load an implementation module.

In most situations, most programs don't really care what specific
implementation module they use, if they aren't already committed to some other
event system and also using C<Future::IO> alongside it. This method allows
programs to offload the decision-making about which specific implementations
to try to load, to C<Future::IO> itself.

This method works by attempting a few different strategies to determine the
"best" implementation to use. It maintains a list of the currently-known CPAN
modules which provide implementations, and attempts them in a given preference
order.

The environment variable C<PERL_FUTURE_IO_IMPL> offers further control of the
behaviour of this method. Its value should be a comma-separated list of
implementation names to be attempted, in preference to any of the others.
Names prefixed with a hyphen will be skipped entirely by any attempt.

=over 4

=item 1.

First, if C<PERL_FUTURE_IO_IMPL> is set, any of the names given are tried, in
order.

=item 2.

Then any of the modules that attempt to wrap other event systems such as L<UV>
or L<Glib> are attempted if it is detected that the other event system is
already loaded.

=item 3.

If none of these were successful, next it attempts any OS-specific modules
based on the OS name (given by C<$^O>).

=item 4.

Finally, a list of other generic modules is attempted, which also includes
any of the wrapper implementations that can be started independently.

=back

For more details, consult the implementation code in this module to find the
current list of known modules and the order they are attempted in.

=cut

# Try to account for every Future::IO::Impl::* module on CPAN
#
# Purposely omitting:
#    Future::IO::Impl::Tickit - requires a $tickit instance to work

my @IMPLS_WRAPPER = (
   "UV",
   "Glib",
   [ IOAsync => "IO::Async::Loop" ],
   "POE",
   "AnyEvent",
);

my %IMPLS_FOR_OS = (
   linux => [qw( Uring )],
   # TODO: other OSes?
);

my @IMPLS_GENERIC = (qw(
   Ppoll
   UV
   Glib
));

sub load_best_impl
{
   shift;

   my @prefer;
   my %veto;

   foreach ( split m/,/, $ENV{PERL_FUTURE_IO_IMPL} // "" ) {
      if( s/^-// ) {
         $veto{$_} = 1;
      }
      else {
         push @prefer, $_;
      }
   }

   foreach my $impl ( @prefer ) {
      $veto{$impl} and next;
      Future::IO->try_load_impl( $impl ) and return 1;
   }

   # First, load a wrapper impl if the wrapped system is already loaded
   foreach ( @IMPLS_WRAPPER ) {
      my ( $impl, $package ) = ref $_ ? @$_ : ( $_, $_ );
      $veto{$impl} and next;
      eval { $package->VERSION(0) } or next;

      Future::IO->try_load_impl( $impl ) and return 1;
   }

   # OK, maybe we can find a good impl for this particular OS
   foreach my $impl ( @{ $IMPLS_FOR_OS{$^O} || [] } ) {
      $veto{$impl} and next;
      Future::IO->try_load_impl( $impl ) and return 1;
   }

   # Failing all of that, try the generic ones
   foreach my $impl ( @IMPLS_GENERIC ) {
      $veto{$impl} and next;
      Future::IO->try_load_impl( $impl ) and return 1;
   }

   die "Unable to find a usable Future::IO::Impl subclass\n";
}

=head2 HAVE_MULTIPLE_FILEHANDLES

   $has = Future::IO->HAVE_MULTIPLE_FILEHANDLES;

I<Since version 0.11.>

Returns true if the underlying IO implementation actually supports multiple
filehandles. The default minimal internal implementation used not to support
this, but I<since version 0.21> it now does; so this method always returns
true.

=cut

sub HAVE_MULTIPLE_FILEHANDLES
{
   return ( $IMPL //= "Future::IO::_DefaultImpl" )->HAVE_MULTIPLE_FILEHANDLES;
}

package
   Future::IO::_DefaultImpl;
use base qw( Future::IO::ImplBase );
use Carp;

use IO::Poll qw( POLLIN POLLOUT POLLPRI );
use Struct::Dumb qw( readonly_struct );
use Time::HiRes qw( time );

readonly_struct Poller => [qw( fh events f )];
readonly_struct Alarm => [qw( time f )];

use constant HAVE_MULTIPLE_FILEHANDLES => 1;

sub alarm
{
   my $class = shift;
   return $class->_done_at( shift );
}

sub sleep
{
   my $class = shift;
   return $class->_done_at( time() + shift );
}

sub poll
{
   my $class = shift;
   my ( $fh, $events ) = @_;

   my $f = Future::IO::_DefaultImpl::F->new;

   push @pollers, Poller( $fh, $events, $f );

   $f->on_cancel( sub {
      my $f = shift;

      my $idx = 0;
      $idx++ while $idx < @pollers and $pollers[$idx]->f != $f;

      splice @pollers, $idx, 1, ();
   });

   return $f;
}

sub waitpid
{
   croak "This implementation cannot handle waitpid";
}

sub _done_at
{
   shift;
   my ( $time ) = @_;

   my $f = Future::IO::_DefaultImpl::F->new;

   # TODO: Binary search
   my $idx = 0;
   $idx++ while $idx < @alarms and $alarms[$idx]->time < $time;

   splice @alarms, $idx, 0, Alarm( $time, $f );

   $f->on_cancel( sub {
      my $self = shift;

      my $idx = 0;
      $idx++ while $idx < @alarms and $alarms[$idx]->f != $f;

      splice @alarms, $idx, 1, ();
   } );

   return $f;
}

package # hide
   Future::IO::_DefaultImpl::F;
use base qw( Future );
use IO::Poll qw( POLLIN POLLOUT POLLPRI );
use Time::HiRes qw( time );

sub _await_once
{
   die "Cowardly refusing to sit idle and do nothing" unless @pollers || @alarms;

   my $rvec = '';
   my $wvec = '';
   my $evec = '';

   foreach my $p ( @pollers ) {
      my $fileno = $p->fh->fileno;

      vec( $rvec, $fileno, 1 ) = 1 if $p->events & POLLIN;
      vec( $wvec, $fileno, 1 ) = 1 if $p->events & POLLOUT;
      vec( $evec, $fileno, 1 ) = 1 if $p->events & POLLPRI;
   }

   # If we always select() then problematic platforms like MSWin32 would
   # always break. Instead, we'll only select() if we're waiting on alarms, or
   # both POLLIN and POLLOUT, or POLLPRI. If not we'll just presume the one
   # operation we're waiting for is definitely ready right now.
   my $do_select = @alarms ||
      ( $rvec ne '' and $wvec ne '' ) ||
      ( $evec ne '' );

   if( $do_select ) {
      my $maxwait;
      $maxwait = $alarms[0]->time - time() if @alarms;

      my $ret = select( $rvec, $wvec, $evec, $maxwait );
   }
   # else just presume it's ready

   # Perl doesn't have an easy construction for iterating an array possibly
   # splicing as you go...
   for ( my $idx = 0; $idx < @pollers; ) {
      my $p = $pollers[$idx];

      my $fh = $p->fh;
      my $fileno = $fh->fileno;

      my $was_blocking;
      $was_blocking = $fh->blocking(1) if !$do_select;

      my $revents = 0;
      $revents |= POLLIN  if vec( $rvec, $fileno, 1 );
      $revents |= POLLOUT if vec( $wvec, $fileno, 1 );
      $revents |= POLLPRI if vec( $evec, $fileno, 1 );
      $revents &= $p->events;

      $revents or $idx++, next;

      splice @pollers, $idx, 1, ();
      $p->f->done( $revents );

      $fh->blocking(0) if !$do_select and !$was_blocking;
   }

   my $now = time();
   while( @alarms and $alarms[0]->time <= $now ) {
      ( shift @alarms )->f->done;
   }
}

=head2 await

   $f = $f->await;

I<Since version 0.07.>

Blocks until this future is ready (either by success or failure). Does not
throw an exception if failed.

=cut

sub await
{
   my $self = shift;
   _await_once until $self->is_ready;
   return $self;
}

=head1 THE C<$IMPL> VARIABLE

I<Since version 0.02.>

As an alternative to setting an implementation by using L<override_impl>, a
package variable is also available that allows modules such as event systems
to opportunistically provide an implementation without needing to depend on
the module, or loading it C<require>. Simply directly set that package
variable to the name of an implementing package or an object instance.

Additionally, implementors may use a name within the C<Future::IO::Impl::>
namespace, suffixed by the name of their event system.

For example, something like the following code arrangement is recommended.

   package Future::IO::Impl::BananaLoop;

   {
      no warnings 'once';
      ( $Future::IO::IMPL //= __PACKAGE__ ) eq __PACKAGE__ or
         warn "Unable to set Future::IO implementation to " . __PACKAGE__ .
            " as it is already $Future::IO::IMPL\n";
   }

   sub poll
   {
      ...
   }

   sub sleep
   {
      ...
   }

   sub sysread
   {
      ...
   }

   sub syswrite
   {
      ...
   }

   sub waitpid
   {
      ...
   }

Optionally, you can also implement L</sysread_exactly> and
L</syswrite_exactly>:

   sub sysread_exactly
   {
      ...
   }

   sub syswrite_exactly
   {
      ...
   }

If not, they will be emulated by C<Future::IO> itself, making multiple calls
to the non-C<_exactly> versions.

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
